/*
 * Decompiled with CFR 0.152.
 */
package org.broad.igv.feature;

import com.google.common.base.Objects;
import com.google.common.collect.Table;
import com.google.common.collect.TreeBasedTable;
import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonNull;
import com.google.gson.JsonObject;
import com.google.gson.JsonParseException;
import com.google.gson.JsonParser;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.InputMismatchException;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.apache.log4j.Logger;
import org.bouncycastle.asn1.ASN1Encodable;
import org.bouncycastle.asn1.ASN1InputStream;
import org.bouncycastle.asn1.ASN1Object;
import org.bouncycastle.asn1.ASN1Primitive;
import org.bouncycastle.asn1.ASN1Set;
import org.bouncycastle.asn1.ASN1String;
import org.bouncycastle.asn1.ASN1TaggedObject;
import org.bouncycastle.asn1.DERInteger;
import org.broad.igv.feature.AminoAcid;
import org.broad.igv.feature.AminoAcidSequence;
import org.broad.igv.feature.Strand;
import org.broad.igv.util.ParsingUtils;

public class AminoAcidManager {
    private static final Logger log = Logger.getLogger(AminoAcidManager.class);
    private static final String AANameFilePath = "resources/AANamesTable.txt";
    private static final Map<String, AminoAcid> AANameMap = new HashMap<String, AminoAcid>(20);
    private static final String[] BASE_SEQUENCES = new String[]{"TTTTTTTTTTTTTTTTCCCCCCCCCCCCCCCCAAAAAAAAAAAAAAAAGGGGGGGGGGGGGGGG", "TTTTCCCCAAAAGGGGTTTTCCCCAAAAGGGGTTTTCCCCAAAAGGGGTTTTCCCCAAAAGGGG", "TCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAG"};
    static final String DEFAULT_CODON_TABLE_PATH = "resources/geneticCode.json";
    static final String DEFAULT_TRANS_TABLE_PATH = "resources/defaultTranslationTables.json";
    public static final int STANDARD_TABLE_ID = 1;
    private static final String DEFAULT_CHROMO_KEY = "default";
    private LinkedHashMap<CodonTableKey, CodonTable> allCodonTables = new LinkedHashMap(20);
    private CodonTable currentCodonTable;
    private static Table<String, String, CodonTableKey> genomeChromoTable = TreeBasedTable.create();
    private static AminoAcidManager instance;

    private AminoAcidManager() {
        AminoAcidManager.initAANameMap();
        try {
            AminoAcidManager.loadDefaultTranslationTables();
        }
        catch (JsonParseException e) {
            log.error((Object)e);
        }
    }

    public static AminoAcidManager getInstance() {
        if (instance == null) {
            try {
                AminoAcidManager newInstance = new AminoAcidManager();
                newInstance.loadCodonTables(DEFAULT_CODON_TABLE_PATH);
                instance = newInstance;
            }
            catch (IOException e) {
                AminoAcidManager.handleExceptionLoading(e);
            }
            catch (JsonParseException e) {
                AminoAcidManager.handleExceptionLoading((Exception)((Object)e));
            }
        }
        return instance;
    }

    public static AminoAcidManager resetToDefaultCodonTables() {
        instance = null;
        return AminoAcidManager.getInstance();
    }

    private static void handleExceptionLoading(Exception e) {
        log.error((Object)e);
        if (instance == null) {
            throw new IllegalStateException("No codon table present, and error loading resources/geneticCode.json", e);
        }
    }

    synchronized void clear() {
        this.allCodonTables.clear();
        this.currentCodonTable = null;
    }

    public boolean setCodonTable(String codonTablePath, int id) {
        CodonTableKey key = new CodonTableKey(codonTablePath, id);
        return this.setCodonTable(key);
    }

    public boolean setCodonTable(CodonTableKey key) {
        if (this.allCodonTables.containsKey(key)) {
            this.currentCodonTable = this.allCodonTables.get(key);
            return true;
        }
        return false;
    }

    public AminoAcid getAminoAcid(String codon) {
        return this.currentCodonTable.getAminoAcid(codon);
    }

    List<AminoAcid> getAminoAcids(Strand direction, String sequence) {
        int readLength = sequence.length() / 3;
        ArrayList<AminoAcid> acids = new ArrayList<AminoAcid>(readLength);
        for (int i = 0; i <= sequence.length() - 3; i += 3) {
            String codon = sequence.substring(i, i + 3).toUpperCase();
            if (direction == Strand.NEGATIVE) {
                codon = AminoAcidManager.getNucleotideComplement(codon);
            }
            AminoAcid aa = this.currentCodonTable.getAminoAcid(codon);
            acids.add(aa);
        }
        return acids;
    }

    public synchronized AminoAcidSequence getAminoAcidSequence(Strand strand, int startPosition, byte[] seqBytes) {
        if (seqBytes == null) {
            return null;
        }
        String nucSequence = new String(seqBytes);
        List<AminoAcid> acids = this.getAminoAcids(strand, nucSequence);
        return new AminoAcidSequence(strand, startPosition, acids, this.currentCodonTable.getKey());
    }

    public synchronized AminoAcidSequence getAminoAcidSequence(Strand strand, int startPosition, String nucleotides) {
        if (nucleotides == null) {
            return null;
        }
        List<AminoAcid> acids = this.getAminoAcids(strand, nucleotides);
        return new AminoAcidSequence(strand, startPosition, acids, this.currentCodonTable.getKey());
    }

    public static AminoAcid getAminoAcidByName(String name) {
        AminoAcidManager.initAANameMap();
        AminoAcid aa = AANameMap.get(name);
        if (aa == null) {
            aa = AminoAcid.NULL_AMINO_ACID;
        }
        return aa;
    }

    public static String getNucleotideComplement(String sequence) {
        char[] complement = new char[sequence.length()];
        int jj = complement.length;
        block6: for (int ii = 0; ii < sequence.length(); ++ii) {
            char c = sequence.charAt(ii);
            --jj;
            switch (c) {
                case 'T': 
                case 't': {
                    complement[jj] = 65;
                    continue block6;
                }
                case 'A': 
                case 'a': {
                    complement[jj] = 84;
                    continue block6;
                }
                case 'C': 
                case 'c': {
                    complement[jj] = 71;
                    continue block6;
                }
                case 'G': 
                case 'g': {
                    complement[jj] = 67;
                    continue block6;
                }
                default: {
                    complement[jj] = c;
                }
            }
        }
        return new String(complement);
    }

    public Set<String> getMappingSNPs(String codon, AminoAcid mutAA) {
        HashSet<String> mapSNPs = new HashSet<String>();
        Set<String> SNPs = AminoAcidManager.getAllSNPs(codon);
        for (String modCodon : SNPs) {
            if (!this.currentCodonTable.getAminoAcid(modCodon).equalsByName(mutAA.getShortName())) continue;
            mapSNPs.add(modCodon);
        }
        return mapSNPs;
    }

    public static Set<String> getAllSNPs(String sequence) {
        HashSet<String> SNPs = new HashSet<String>();
        char[] bps = "ATGC".toCharArray();
        char[] orig = sequence.toCharArray();
        for (int loc = 0; loc < orig.length; ++loc) {
            char[] mod = (char[])orig.clone();
            for (char bp : bps) {
                if (bp == orig[loc]) continue;
                mod[loc] = bp;
                SNPs.add(new String(mod));
            }
        }
        return SNPs;
    }

    synchronized void loadCodonTables(String codonTablesPath) throws IOException, JsonParseException {
        LinkedHashMap<CodonTableKey, CodonTable> newCodonTables = new LinkedHashMap<CodonTableKey, CodonTable>(20);
        CodonTable defaultCodonTable = null;
        InputStream is = AminoAcidManager.class.getResourceAsStream(codonTablesPath);
        if (is == null) {
            is = ParsingUtils.openInputStream(codonTablesPath);
        }
        if (codonTablesPath.endsWith(".json")) {
            JsonObject allData = AminoAcidManager.readJSONFromStream(is);
            int defaultId = -1;
            defaultId = allData.get("defaultid").getAsInt();
            JsonArray codonArray = allData.get("Genetic-code-table").getAsJsonArray();
            if (codonArray.size() == 0) {
                throw new JsonParseException("JSON File has empty array for Genetic-code-table");
            }
            for (int ca = 0; ca < codonArray.size(); ++ca) {
                CodonTable curTable = CodonTable.createFromJSON(codonTablesPath, codonArray.get(ca).getAsJsonObject());
                newCodonTables.put(curTable.getKey(), curTable);
                if (defaultCodonTable != null && curTable.getId() != defaultId) continue;
                defaultCodonTable = curTable;
            }
        } else if (codonTablesPath.endsWith(".asn1") || codonTablesPath.endsWith(".val")) {
            ASN1InputStream ASNis = new ASN1InputStream(is);
            ASN1Primitive obj = ASNis.readObject();
            ASN1Set set = (ASN1Set)obj;
            ASN1Encodable[] codonArray = set.toArray();
            if (codonArray.length == 0) {
                throw new RuntimeException("ASN1 File has empty array for Genetic-code-table");
            }
            for (ASN1Encodable aCodonArray : codonArray) {
                CodonTable curTable = CodonTable.createFromASN1(codonTablesPath, aCodonArray);
                newCodonTables.put(curTable.getKey(), curTable);
                if (defaultCodonTable != null) continue;
                defaultCodonTable = curTable;
            }
        } else {
            throw new IllegalArgumentException("Unknown file type, must be .json or .asn1");
        }
        this.allCodonTables.putAll(newCodonTables);
        this.currentCodonTable = defaultCodonTable;
    }

    private static JsonObject readJSONFromStream(InputStream is) {
        BufferedReader reader = new BufferedReader(new InputStreamReader(is));
        JsonParser parser = new JsonParser();
        return parser.parse((Reader)reader).getAsJsonObject();
    }

    private static synchronized void initAANameMap() {
        if (!AANameMap.isEmpty()) {
            return;
        }
        try {
            String nextLine;
            InputStream is = AminoAcidManager.class.getResourceAsStream(AANameFilePath);
            if (is == null) {
                return;
            }
            BufferedReader reader = new BufferedReader(new InputStreamReader(is));
            while ((nextLine = reader.readLine()) != null) {
                String[] tokens;
                if (nextLine.startsWith("#") || (tokens = nextLine.split("\t")).length != 3) continue;
                String fullName = tokens[0].trim();
                String shortName = tokens[1].trim();
                String symbol = tokens[2].trim();
                assert (symbol.length() == 1);
                AminoAcid aa = new AminoAcid(fullName, shortName, symbol.charAt(0));
                for (String sym : new String[]{fullName, shortName, symbol}) {
                    if (AANameMap.containsKey(sym)) continue;
                    AANameMap.put(sym, aa);
                }
            }
        }
        catch (IOException ex) {
            log.error((Object)ex);
            throw new RuntimeException(ex);
        }
    }

    public Collection<CodonTable> getAllCodonTables() {
        return Collections.unmodifiableCollection(this.allCodonTables.values());
    }

    public CodonTable getCodonTable() {
        return this.currentCodonTable;
    }

    private static void loadDefaultTranslationTables() throws JsonParseException {
        InputStream is = AminoAcidManager.class.getResourceAsStream(DEFAULT_TRANS_TABLE_PATH);
        JsonObject allData = AminoAcidManager.readJSONFromStream(is);
        JsonArray organisms = allData.get("organisms").getAsJsonArray();
        for (int ind = 0; ind < organisms.size(); ++ind) {
            JsonObject obj = organisms.get(ind).getAsJsonObject();
            String genomeId = obj.get("genomeId").getAsString();
            String codonTablePath = DEFAULT_CODON_TABLE_PATH;
            try {
                JsonElement tmpPath = obj.get("codonTablePath");
                if (tmpPath != null && tmpPath != JsonNull.INSTANCE && tmpPath instanceof String) {
                    codonTablePath = (String)tmpPath;
                }
            }
            catch (JsonParseException e) {
                log.error((Object)("No codon table path found in resources/defaultTranslationTables.json. Using default: " + codonTablePath));
            }
            JsonObject chromosomes = obj.get("chromosomes").getAsJsonObject();
            for (Map.Entry entry : chromosomes.entrySet()) {
                String chromoName = (String)entry.getKey();
                int id = ((JsonElement)entry.getValue()).getAsInt();
                CodonTableKey key = new CodonTableKey(codonTablePath, id);
                genomeChromoTable.put((Object)genomeId, (Object)chromoName, (Object)key);
            }
        }
    }

    public static class CodonTable {
        private final CodonTableKey key;
        private final List<String> names;
        private final Set<AminoAcid> starts;
        private final Map<String, AminoAcid> codonMap;

        public AminoAcid getAminoAcid(String codon) {
            if (codon.length() != 3) {
                throw new IllegalArgumentException("Codon must be length 3: " + codon);
            }
            AminoAcid aa = this.codonMap.get(codon);
            if (aa == null) {
                return AminoAcid.NULL_AMINO_ACID;
            }
            return aa;
        }

        private CodonTable(String path, int id, List<String> names, Set<AminoAcid> starts, Map<String, AminoAcid> codonMap) {
            this.key = new CodonTableKey(path, id);
            this.names = Collections.unmodifiableList(names);
            this.starts = Collections.unmodifiableSet(starts);
            this.codonMap = Collections.unmodifiableMap(codonMap);
        }

        private static CodonTable createFromJSON(String sourcePath, JsonObject jsonObject) throws JsonParseException {
            int id = jsonObject.get("id").getAsInt();
            JsonArray jsonnames = jsonObject.get("name").getAsJsonArray();
            ArrayList<String> names = new ArrayList<String>(jsonnames.size());
            for (int nn = 0; nn < jsonnames.size(); ++nn) {
                names.add(jsonnames.get(nn).getAsString());
            }
            String aas = jsonObject.get("ncbieaa").getAsString();
            String startString = jsonObject.get("sncbieaa").getAsString();
            return CodonTable.build(sourcePath, id, names, aas, startString);
        }

        private static CodonTable createFromASN1(String sourcePath, ASN1Encodable asn1Encodable) throws IOException {
            byte[] data = asn1Encodable.toASN1Primitive().getEncoded();
            ASN1InputStream iASNis = new ASN1InputStream(data);
            ASN1Primitive prim = iASNis.readObject();
            ASN1Set iset = (ASN1Set)prim;
            ASN1TaggedObject[] taggedObjects = CodonTable.getTaggedObjects(iset.toArray());
            int index = 0;
            int tagNo = taggedObjects[index].getTagNo();
            ArrayList<String> names = new ArrayList<String>(2);
            while (tagNo == 0) {
                names.add(CodonTable.getAsString((ASN1Object)taggedObjects[index].getObject()));
                tagNo = taggedObjects[++index].getTagNo();
            }
            int id = ((DERInteger)taggedObjects[index++].getObject()).getValue().intValue();
            String aas = CodonTable.getAsString((ASN1Object)taggedObjects[index++].getObject());
            String startString = CodonTable.getAsString((ASN1Object)taggedObjects[index++].getObject());
            return CodonTable.build(sourcePath, id, names, aas, startString);
        }

        private static String getAsString(ASN1Object object) {
            return ((ASN1String)object).getString();
        }

        private static ASN1TaggedObject[] getTaggedObjects(ASN1Encodable[] encodables) {
            ASN1TaggedObject[] taggedObjects = new ASN1TaggedObject[encodables.length];
            for (int ii = 0; ii < encodables.length; ++ii) {
                taggedObjects[ii] = (ASN1TaggedObject)encodables[ii];
            }
            return taggedObjects;
        }

        private static CodonTable build(String sourcePath, int id, List<String> names, String aas, String startString) {
            String base1 = BASE_SEQUENCES[0];
            String base2 = BASE_SEQUENCES[1];
            String base3 = BASE_SEQUENCES[2];
            CodonTable.checkLengths(base1, base2, base3, aas, startString);
            HashMap<String, AminoAcid> codonMap = new HashMap<String, AminoAcid>(aas.length());
            HashSet<AminoAcid> starts = new HashSet<AminoAcid>(aas.length());
            for (int cc = 0; cc < aas.length(); ++cc) {
                String codon = base1.substring(cc, cc + 1) + base2.substring(cc, cc + 1) + base3.substring(cc, cc + 1);
                AminoAcid aa = (AminoAcid)AANameMap.get(aas.substring(cc, cc + 1));
                codonMap.put(codon, aa);
                if (startString.charAt(cc) != 'M') continue;
                starts.add(aa);
            }
            return new CodonTable(sourcePath, id, names, starts, codonMap);
        }

        private static void checkLengths(String ... values) {
            int length = values[0].length();
            assert (length == 64);
            for (int v = 1; v < values.length; ++v) {
                if (values[v].length() == length) continue;
                String msg = "Amino acid and codon strings must all be the same length.";
                msg = msg + "Expected length " + length + ", found length " + values[v].length();
                throw new InputMismatchException(msg);
            }
        }

        public int getId() {
            return this.key.id;
        }

        public String getDisplayName() {
            return this.names.get(0);
        }

        public Set<AminoAcid> getStarts() {
            return this.starts;
        }

        Map<String, AminoAcid> getCodonMap() {
            return this.codonMap;
        }

        public boolean equals(Object object) {
            if (object instanceof CodonTable) {
                CodonTable other = (CodonTable)object;
                return Objects.equal((Object)this.key, (Object)other.key) && Objects.equal(this.names, other.names) && Objects.equal(this.starts, other.starts) && Objects.equal(this.codonMap, other.codonMap);
            }
            return false;
        }

        public int hashCode() {
            return Objects.hashCode((Object[])new Object[]{this.key.id, this.key.sourcePath, this.names, this.starts, this.codonMap});
        }

        public CodonTableKey getKey() {
            return this.key;
        }
    }

    public static class CodonTableKey {
        private final String sourcePath;
        private final int id;

        private CodonTableKey(String sourcePath, int id) {
            this.sourcePath = sourcePath;
            this.id = id;
        }

        public boolean equals(Object object) {
            if (object instanceof CodonTableKey) {
                CodonTableKey other = (CodonTableKey)object;
                return this.id == other.id && Objects.equal((Object)this.sourcePath, (Object)other.sourcePath);
            }
            return false;
        }

        public int hashCode() {
            return Objects.hashCode((Object[])new Object[]{this.sourcePath, this.id});
        }

        public int getId() {
            return this.id;
        }
    }
}

